<?php

namespace App\Http\Resources\API\MarketPlace;

use App\Domain\Order\Models\Order;
use App\Domain\Order\OrderStatus\Enums\AppointmentOrderStatusEnum;
use App\Domain\Order\OrderStatus\Enums\MarketPlaceOrderStatusEnum;
use App\Domain\Order\OrderStatus\Enums\SparePartOrderStatusEnum;
use App\Domain\Order\OrderStatus\Enums\WenchOrderStatusEnum;
use App\Http\Resources\API\Core\BranchResource;
use App\Http\Resources\API\Order\OfferOrderResource;
use App\Http\Resources\API\Order\ServiceOrderResource;
use App\Http\Resources\API\Order\SparePartOrderResource;
use App\Http\Resources\API\Order\VoucherResource;
use App\Support\Traits\WithPagination;
use Carbon\Carbon;
use Illuminate\Http\Resources\Json\JsonResource;

class OrderDetailsResource extends JsonResource
{
    use WithPagination;

    /**
     * Transform the resource into an array.
     *
     * @param \Illuminate\Http\Request $request
     * @return array|\Illuminate\Contracts\Support\Arrayable|\JsonSerializable
     */
    public function toArray($request)
    {
        $status = match (true) {
            ($this?->type == 'appointment' && $this->status) =>
            AppointmentOrderStatusEnum::{$this->status}()->value,
            ($this?->type == 'spare_part' && $this->status) =>
            SparePartOrderStatusEnum::{$this->status}()->value,
            ($this?->type == 'wench' && $this->status) =>
            WenchOrderStatusEnum::{$this->status}()->value,
            ($this?->type == 'marketplace' && $this->status) =>
            MarketPlaceOrderStatusEnum::{$this->status}()->value,
            default => __('Status None'),
        };
        $order = Order::find($this->id);
        $order_attributes = [
            [
                'label' => __('Total Order'),
                'value' => $this->sub_total + $this->discount_amount + $this->promo_discount_amount . ' ' . __('SAR')
            ],
        ];
        if ($this->discount_amount || $this->promo_discount_amount) {
            $order_attributes[] = [
                'label' => __('Discount Amount'),
                'value' => $this->discount_amount + $this->promo_discount_amount . ' ' . __('SAR')
            ];
            $order_attributes[] = [
                'label' => __('Total Order After discount'),
                'value' => $this->sub_total . ' ' . __('SAR')
            ];
        }
        $order_attributes[] =
            [
                'label' => __('Vat'),
                'value' => $this->vat_amount . ' ' . __('SAR')
            ];
        $order_attributes[] = [
            'label' => __('Final Amount'),
            'value' => $this->total_amount . ' ' . __('SAR')
        ];

        return [
            'id' => $this->id,
            'type' => $this->type,
            'status' => $this->status,
            'status_value' => $status,
            'sub_total' => number_format($this->sub_total, 2, '.', ''),
            'vat_amount' => number_format($this->vat_amount, 2, '.', ''),
            'total_amount' => number_format($this->total_amount, 2, '.', ''),
            'amount_to_pay' => number_format($this->amountToPay(), 2, '.', ''),

            'spare_parts' => SparePartOrderResource::collection($this->spareParts),
            'offers' => OfferOrderResource::collection($this->offers),
            'services' => ServiceOrderResource::collection($this->services),
            'voucher' => new VoucherResource($this->voucher, $this->id),
            'reservation_time' => $this->reservation_time ? Carbon::parse($this->reservation_time)->translatedFormat('D, M j, Y g:i A') : null,
            'branch' => new BranchResource($this->branch),
            'customer_note' => $this->customer_note,
            'tamara_pay' => true,
            'visa_pay' => true,
            'mada_pay' => true,
            'apple_pay' => true,
            'cod_pay' => false,
            'allowed_payment_methods' => [
                'tamara',
                'visa',
                'master',
                'mada',
                'apple_pay',
                'cod'
            ],
            'created_at' => $this->created_at->translatedFormat('D, M j, Y g:i A'),
            'order_attributes' => $order_attributes
        ];
    }

}
