<?php

namespace App\Jobs;

use App\Domain\Link\Models\Difference;
use App\Domain\Link\Models\Link;
use App\Services\ScreenshotService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class TakeScreenshotJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public $link;

    /**
     * Create a new job instance.
     *
     * @param Link $link
     * @return void
     */
    public function __construct(Link $link)
    {
        $this->link = $link;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        $url = $this->link->url;

        // Paths for new and old screenshots
        $newScreenshotPath = storage_path('app/public/screenshots/') . 'new_screenshot_' . $this->link->id . '_' . time() . '.png';
        $oldScreenshotPath = storage_path('app/public/screenshots/') . 'old_screenshot_' . $this->link->id . '.png';

        // Take the new screenshot
        ScreenshotService::takeScreenshot($url, $newScreenshotPath);

        // Check if there's an existing "new" screenshot path in the link
        if ($this->link->new_screenshot_path && file_exists($this->link->new_screenshot_path)) {
            // Move the current "new" screenshot to "old"
            rename($this->link->new_screenshot_path, $oldScreenshotPath);

            // Update the link with the old screenshot path
            $this->link->old_screenshot_path = $oldScreenshotPath;
        }

        // Compare the old and new screenshots if both exist
        if (file_exists($oldScreenshotPath) && file_exists($newScreenshotPath)) {
            $difference = ScreenshotService::compareImagesWithIntervention($oldScreenshotPath, $newScreenshotPath, 10); // tolerance of 10

            if ($difference > 0) {
                // Log and save the difference in the database
                Difference::create([
                    'link_id' => $this->link->id,
                    'difference_score' => $difference,
                ]);

                \Log::info("Difference found for link: " . $url . " with difference score: " . $difference);
            } else {
                // No difference found; optionally delete the new screenshot if it matches the old one
                \Log::info("No difference found for link: " . $url);
            }
        }

        // Update the link's new screenshot path and save it
        $this->link->new_screenshot_path = $newScreenshotPath;
        $this->link->save();
    }

}
